# decorators.py
import json
from django.http import JsonResponse
from django.http import HttpResponseNotAllowed
from django.shortcuts import redirect



#  ============================= auth middleware =======================================
def auth_required(view_func):
    def wrapper(request, *args, **kwargs):
        if not request.user.is_authenticated:
            return redirect('login')  # Replace 'login' with your login URL

        return view_func(request, *args, **kwargs)

    return wrapper


def guest_only(view_func):
    def wrapper(request, *args, **kwargs):
        if request.user.is_authenticated:
            return redirect('home')  # Replace 'login' with your login URL

        return view_func(request, *args, **kwargs)

    return wrapper


def company_staff_required(view_func):
    def wrapper(request, *args, **kwargs):
        # Check if the user is authenticated
        if not request.user.is_authenticated:
            return redirect('panel_login')
        # Check if the user is a panel staff
        if not request.user.is_company_staff:
            # You can customize this redirect to a page indicating insufficient permissions
            return redirect('panel_login')

        return view_func(request, *args, **kwargs)

    return wrapper

def superuser_required(view_func):
    def wrapper(request, *args, **kwargs):
        # Check if the user is authenticated
        if not request.user.is_authenticated:
            return redirect('panel_login')
        # Check if the user is a panel staff
        if not request.user.is_superuser:
            # You can customize this redirect to a page indicating insufficient permissions
            return redirect('panel_login')

        return view_func(request, *args, **kwargs)

    return wrapper

def manager_required(view_func):
    def wrapper(request, *args, **kwargs):
        # Check if the user is authenticated
        if not request.user.is_authenticated:
            return redirect('panel_login')
        # Check if the user is a panel staff
        if not request.user.is_manager:
            # You can customize this redirect to a page indicating insufficient permissions
            return redirect('panel_login')

        return view_func(request, *args, **kwargs)

    return wrapper

def assitant_manager_required(view_func):
    def wrapper(request, *args, **kwargs):
        # Check if the user is authenticated
        if not request.user.is_authenticated:
            return redirect('panel_login')
        # Check if the user is a panel staff
        if not request.user.is_assitant_manager:
            # You can customize this redirect to a page indicating insufficient permissions
            return redirect('panel_login')

        return view_func(request, *args, **kwargs)

    return wrapper
# ================================= request method middleware ====================================

def process_json_data(view_func):
    def wrapper(request, *args, **kwargs):
        if request.method == 'POST':
            try:
                request_data = json.loads(request.body.decode('utf-8'))
                request.json_data = request_data
            except json.JSONDecodeError:
                return JsonResponse({'success': False, 'errors': 'Invalid JSON data'}, status=400)

        return view_func(request, *args, **kwargs)
    
    return wrapper

def allow_only_GET(view_func):
    def _wrapped_view(request, *args, **kwargs):
        if request.method != 'GET':
            return HttpResponseNotAllowed(['GET'], content="<html><body><h1>Method Not Allowed</h1><p>This endpoint only allows GET requests.</p></body></html>")
        return view_func(request, *args, **kwargs)

    return _wrapped_view

def allow_only_POST(view_func):
    def _wrapped_view(request, *args, **kwargs):
        if request.method != 'POST':
            return HttpResponseNotAllowed(['POST'], content="<html><body><h1>Method Not Allowed</h1><p>This endpoint only allows POST requests.</p></body></html>")
        return view_func(request, *args, **kwargs)

    return _wrapped_view

def allow_only_content_JSON(view_func):
    def _wrapped_view(request, *args, **kwargs):

        if not request.content_type == 'application/json':
            return HttpResponseNotAllowed(['application/json'] ,content="<html><body><h1>Not Allowed</h1><p>This endpoint only allows JSON POST requests.</p></body></html>")

        return view_func(request, *args, **kwargs)

    return _wrapped_view


## **************************************  mixins.py ******************************************************

from django import forms

class CustomFieldMixin(forms.ModelForm):
    img_class = 'img-class'
    choice_class = 'choice-class'

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        for field_name, field in self.fields.items():
            if isinstance(field, forms.ImageField):
                field.widget.attrs.update({'class': self.img_class})
            if isinstance(field, forms.ChoiceField):
                field.widget.attrs.update({'class': self.choice_class})


# sample use in the form

# class YourForm(CustomFieldMixin, forms.ModelForm):
#     img_class = 'custom-img-class'
#     choice_class = 'custom-choice-class'

#     class Meta:
#         model = YourModel
#         fields = '__all__'

class JsonErrorsMixin:
    def get_errors(self):
        errors_dict = {field: self.errors[field][0] for field in self.errors}
        return {
            'data': errors_dict,
            'has_errors': bool(errors_dict)
        }



## *********************** serilizer helpers ***********************
    

from django.forms.models import model_to_dict
from django.core.serializers.json import DjangoJSONEncoder
from django.db.models.fields.files import ImageFieldFile
class CustomJSONEncoder(DjangoJSONEncoder):
    def default(self, obj):
        if isinstance(obj, ImageFieldFile):
            return obj.url if obj else None
        return super().default(obj)

def serialize_model_data_all(model_queryset, related_fields=None):
    serialized_data = []

    for instance in model_queryset:
        serialized_instance = model_to_dict(instance)
        serialized_instance['created_at'] = instance.created_at.strftime('%Y-%m-%d %H:%M:%S')
        serialized_instance['updated_at'] = instance.updated_at.strftime('%Y-%m-%d %H:%M:%S')

        if related_fields:
            for field in related_fields:
                related_instance = getattr(instance, field)
                if hasattr(related_instance, 'url'):
                    serialized_instance[field] = related_instance.url
                else:
                    serialized_instance[field] = model_to_dict(related_instance)

        serialized_data.append(serialized_instance)

    return serialized_data




def serialize_model_data_single(model_instance, related_fields=None):
    serialized_instance = model_to_dict(model_instance)
    serialized_instance['created_at'] = model_instance.created_at.strftime('%Y-%m-%d %H:%M:%S')
    serialized_instance['updated_at'] = model_instance.updated_at.strftime('%Y-%m-%d %H:%M:%S')

    if related_fields:
        for field in related_fields:
            related_instance = getattr(model_instance, field)
            if hasattr(related_instance, 'url'):
                serialized_instance[field] = related_instance.url if related_instance else None
            else:
                serialized_instance[field] = model_to_dict(related_instance)

    return serialized_instance