from django.db import models
from ckeditor_uploader.fields import RichTextUploadingField
from taggit.managers import TaggableManager
from apps.products.models import *

from django.contrib.auth import get_user_model
# Create your models here.
CustomUser = get_user_model() 



# Create your models here.
class Contact(models.Model):
    name = models.CharField(max_length=100, verbose_name='Full Name')
    email = models.EmailField(verbose_name='Email Address')
    subject = models.CharField(max_length=200, verbose_name='Subject')
    message = models.TextField(verbose_name='Your Message')
    created_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return self.subject

    class Meta:
        verbose_name = 'Contact'
        verbose_name_plural = 'Contacts'


class FAQ(models.Model):
    question = models.CharField(max_length=255, verbose_name='Question')
    answer = models.TextField(verbose_name='Answer')

    def __str__(self):
        return self.question

    class Meta:
        verbose_name = 'FAQ'
        verbose_name_plural = 'FAQs'


# class PrivacyPolicy(models.Model):
#     title = models.CharField(max_length=255, verbose_name='Title')
#     content = RichTextUploadingField(verbose_name='Content')
#     updated_at = models.DateTimeField(auto_now=True)

#     def __str__(self):
#         return self.title

#     class Meta:
#         verbose_name = 'Privacy Policy'
#         verbose_name_plural = 'Privacy Policies'
        


class ContactInformation(models.Model):
    phone_number1 = models.CharField(max_length=15, verbose_name='Primary Phone Number')
    phone_number2 = models.CharField(max_length=15, null=True, blank=True, verbose_name='Secondary Phone Number')
    email1 = models.EmailField(verbose_name='Primary Email')
    email2 = models.EmailField(null=True, blank=True, verbose_name='Secondary Email')
    address1 = models.TextField(max_length=255, verbose_name='Primary Address')
    address2 = models.TextField(max_length=255, null=True, blank=True, verbose_name='Secondary Address')

    class Meta:
        verbose_name = 'Contact Information'
        verbose_name_plural = 'Contact Informations'

    def __str__(self):
        return f"{self.phone_number1}, {self.email1}"
    


class SocialMedia(models.Model):
    FACEBOOK = 'Facebook'
    TWITTER = 'Twitter'
    INSTAGRAM = 'Instagram'
    LINKEDIN = 'LinkedIn'
    SNAPCHAT = 'Snapchat'
    PINTEREST = 'Pinterest'
    TIKTOK = 'TikTok'
    YOUTUBE = 'YouTube'

    SOCIAL_MEDIA_CHOICES = [
        (FACEBOOK, 'Facebook'),
        (TWITTER, 'Twitter'),
        (INSTAGRAM, 'Instagram'),
        (LINKEDIN, 'LinkedIn'),
        (SNAPCHAT, 'Snapchat'),
        (PINTEREST, 'Pinterest'),
        (TIKTOK, 'TikTok'),
        (YOUTUBE, 'YouTube'),
    ]

    name = models.CharField(max_length=50, choices=SOCIAL_MEDIA_CHOICES)
    url = models.URLField()
    icon = models.ImageField(upload_to='social_media_icons/', null=True, blank=True, help_text='Recommended size: 50x50 pixels. This field is optional.')

    class Meta:
        verbose_name = 'Social Media'
        verbose_name_plural = 'Social Medias'

    def __str__(self):
        return self.name
    



#blog post related all models here
class BlogCategory(models.Model):
    name = models.CharField(max_length=255, unique=True)
    slug = models.SlugField(unique=True)


    class Meta:
        ordering = ['name']
        verbose_name = 'Blog Category'
        verbose_name_plural = 'Blog Categories'

    def __str__(self):
        return self.name
    

class Author(models.Model):
    user = models.ForeignKey(CustomUser, on_delete=models.SET_NULL, null=True)
    bio = models.TextField(blank=True, null=True)

    def __str__(self):
        return self.user.name


class AuthorSocialProfile(models.Model):
    FACEBOOK = 'Facebook'
    TWITTER = 'Twitter'
    INSTAGRAM = 'Instagram'
    LINKEDIN = 'LinkedIn'
    SNAPCHAT = 'Snapchat'
    PINTEREST = 'Pinterest'
    TIKTOK = 'TikTok'
    YOUTUBE = 'YouTube'

    SOCIAL_MEDIA_CHOICES = [
        (FACEBOOK, 'Facebook'),
        (TWITTER, 'Twitter'),
        (INSTAGRAM, 'Instagram'),
        (LINKEDIN, 'LinkedIn'),
        (SNAPCHAT, 'Snapchat'),
        (PINTEREST, 'Pinterest'),
        (TIKTOK, 'TikTok'),
        (YOUTUBE, 'YouTube'),
    ]

    author = models.ForeignKey(Author, on_delete=models.CASCADE)
    name = models.CharField(max_length=50, choices=SOCIAL_MEDIA_CHOICES)
    url = models.URLField()

    def __str__(self):
        return f"{self.author.user.name}'s {self.get_name_display()} Profile"
  

class Blog(models.Model):
    title = models.CharField(max_length=255)
    slug = models.SlugField(unique=True)
    content = RichTextUploadingField()
    author = models.ForeignKey(Author, on_delete=models.CASCADE)
    publish_date = models.DateField()
    categories = models.ManyToManyField(BlogCategory)
    featured_image = models.ImageField(upload_to='blog_images/', null=True, blank=True)
    tags = TaggableManager()
    views = models.PositiveIntegerField(default=0)
    is_featured = models.BooleanField(default=False)
    is_recommend = models.BooleanField(default=False)
    is_published = models.BooleanField(default=True)

    def comments(self):
        return Comment.objects.filter(blog=self)

    def author_social_profiles(self):
        return AuthorSocialProfile.objects.filter(author=self.author)

    def __str__(self):
        return self.title
    

class Comment(models.Model):
    blog = models.ForeignKey(Blog, on_delete=models.CASCADE)
    user = models.ForeignKey(CustomUser, on_delete=models.CASCADE)
    text = models.TextField()
    created_at = models.DateTimeField(auto_now_add=True)

    class Meta:
        ordering = ['-created_at']

    def __str__(self):
        return f'{self.user.name} - {self.created_at.strftime("%Y-%m-%d %H:%M:%S")}'


class Reply(models.Model):
    comment = models.ForeignKey(Comment, on_delete=models.CASCADE)
    user = models.ForeignKey(CustomUser, on_delete=models.CASCADE)
    text = models.TextField()
    created_at = models.DateTimeField(auto_now_add=True)
    


class ProductSlider(models.Model):  # app.main
    title = models.CharField(max_length=200)
    short_details = models.CharField(max_length=500)
    price = models.CharField(max_length=100)
    image = models.ImageField(upload_to='product_slider_images/', null=True ,blank=True) 
    product = models.OneToOneField(Product, on_delete=models.RESTRICT)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)